# ------------------------------------------------------------------------------
#
#            Program Dcoumentation Package for the HP-15C Simulator
#
#                          (c) 2017-2018 Torsten Manz
#
# ------------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, see <http://www.gnu.org/licenses/>
#
# ------------------------------------------------------------------------------

package require Tcl 8
package provide prdoc 1.3.0

namespace eval ::prdoc {

  variable CONF
  array set CONF {
    ShowResTab 0
    taghighlight 1
  }

  variable renderHTML 0

  variable LAYOUT
  if {$::tcl_platform(os) == "Darwin"} {
    set LAYOUT(btnwid1) 1
    set LAYOUT(btnwid2) 3
  } else {
    set LAYOUT(btnwid1) 3
    set LAYOUT(btnwid2) 4
  }

  variable DESC
  array set DESC {}
  variable DESC_T
  array set DESC_T {}

  variable MARS
  array set MARKS { L {} R {} F {} }

  # Create fonts and styles for HTML rendering
  set basesize [font actual TkTextFont -size]
# WA-Linux: "font actual" returns "-size 0" for Tk standard fonts
  if {$basesize <= 0} {
    set basesize [expr int([font metrics TkFixedFont -linespace]*0.65)]
  }
  foreach {tt fs wt} {h1 6 bold h2 4 bold h3 3 bold h4 2 bold h5 1 bold h6 0 bold
    strong 0 bold em 0 normal serif 0 normal reg 0 bold keyface 1 bold} {
    font create FnPrDoc$tt -family [font actual TkTextFont -family] \
      -size [expr $basesize+$fs] -weight $wt
  }
  font configure FnPrDocem -slant italic
  font configure FnPrDocserif -family Times -slant italic
  font configure FnPrDocreg -family Times
  font create FnFixbold -family [font actual TkFixedFont -family] \
    -size [expr int($basesize*1.3)] -weight bold

  array set TagStyle {
    h1 {{<h1>} {</h1>} {-font FnPrDoch1 -fore black -back white}}
  	h2 {{<h2>} {</h2>} {-font FnPrDoch2 -fore black -back white}}
  	h3 {{<h3>} {</h3>} {-font FnPrDoch3 -fore black -back white}}
  	h4 {{<h4>} {</h4>} {-font FnPrDoch4 -fore black -back white}}
  	h5 {{<h5>} {</h5>} {-font FnPrDoch5 -fore black -back white}}
  	h6 {{<h6>} {</h6>} {-font FnPrDoch6 -fore black -back white}}
  	bold {{<b>} {</b>} {-font FnPrDocstrong -fore black -back white}}
  	strong {{<strong>} {</strong>} {-font FnPrDocstrong -fore black -back white}}
  	i {{<i>} {</i>} {-font FnPrDocem -fore black -back white}}
  	em {{<em>} {</em>} {-font FnPrDocem -fore black -back white}}
  	code {{<code>} {</code>} {-font TkFixedFont -fore black -back white}}
    ol {{<ol>} {</ol>} {-lmargin1 20} -fore black -back white}
    ul {{<ul>} {</ul>} {-lmargin1 20} -fore black -back white}
    li {{<li>} {</li>} {-lmargin1 20} -fore black -back white}
    a {{<a[^>]+>} {</a>} {-fore blue -back white}}
    KeyLabel {{<span class="HP15CKey">} {</span>} \
      {-font FnPrDockeyface -fore white -back #454545}}
    gKeyLabel {{<span class="HP15CgKeyLabel">} \
      {</span>} {-font FnPrDockeyface -fore #6CB7BD -back #454545}}
    fKeyLabel {{<span class="HP15CfKeyLabel">} \
      {</span>} {-font FnPrDockeyface -fore #E1A83E -back #454545}}
    gKey {{<span class="HP15CgKey">} {</span>} \
      {-font FnPrDockeyface -fore black -back #6CB7BD}}
    fKey {{<span class="HP15CfKey">} {</span>} \
      {-font FnPrDockeyface -fore black -back #E1A83E}}
    register {{<span class="HP15CRegister">} {</span>} \
      {-font FnFixbold -fore black -back white}}
  }

  ttk::style configure strong.TButton -font FnPrDocstrong
  ttk::style configure em.TButton -font FnPrDocserif
  ttk::style configure reg.TButton -font FnPrDocreg
  ttk::style configure fkey.TButton -font FnPrDocstrong -foreground #E1A83E
  ttk::style configure gkey.TButton -font FnPrDocstrong -foreground #6CB7BD

}

# ------------------------------------------------------------------------------
proc ::prdoc::Analyse { prgm } {

  variable MARKS
  array set MARKS { L {} R {} F {} }

  set aregs {}
  set altmp {}
  set nltmp {}

  for {set ii 0} {$ii < [llength $prgm]} {incr ii} {
    if {[regexp {4[45]_([1234]0_)*(48_)*([0-9])$} [lindex $prgm $ii] \
      step oper dec reg]} {
      if {$dec != ""} {incr reg 10}
      lappend MARKS(R) $reg
    } elseif {[regexp {4[45]_([1234]0_)*24$} [lindex $prgm $ii] step]} {
      lappend aregs "(i)"
    } elseif {[regexp {4[45]_([1234]0_)*25$} [lindex $prgm $ii] step]} {
      lappend aregs "I"
    } elseif {[regexp {[23]2_1([1-5])$} [lindex $prgm $ii] step lbl]} {
      lappend altmp -$lbl
    } elseif {[regexp {[23]2_(48_)*([0-9])$} [lindex $prgm $ii] step dec lbl]} {
      if {$dec != ""} {incr lbl 10}
      lappend nltmp $lbl
    } elseif {[regexp {42_21_(48_)*([0-9])$} [lindex $prgm $ii] step dec lbl]} {
      if {$dec != ""} {incr lbl 10}
      lappend nltmp $lbl
    } elseif {[regexp {42_21_1([1-5])$} [lindex $prgm $ii] step lbl]} {
      lappend altmp -$lbl
    }
  }
  set MARKS(L) [concat \
    [lsort -unique -integer -decreasing $altmp] [lsort -unique -integer $nltmp]]

  set MARKS(R) [lsort -unique -integer $MARKS(R)]
  lappend MARKS(R) {*}[join [lsort -unique -dictionary $aregs]]

  for {set ii 0} {$ii < [llength $prgm]} {incr ii} {
    if {[regexp {43_[456]_([0-9])} [lindex $prgm $ii] ign flag]} {
      lappend MARKS(F) $flag
    }
  }
  set MARKS(F) [lsort -unique -integer $MARKS(F)]

}

# ------------------------------------------------------------------------------
proc ::prdoc::Reload {} {

  if {[winfo exists .pdocu]} {
    set geom "+[winfo x .pdocu]+[winfo y .pdocu]"
    destroy .pdocu
    Edit $geom
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::ReDraw {} {

  variable CONF
  variable DESC_T

  if {[winfo exists .pdocu]} {
#     if {$CONF(ShowResTab)} {
#       set geom [winfo geometry .pdocu]
#     } else {
#       set geom "+[winfo x .pdocu]+[winfo y .pdocu]"
#     }
    set geom "[winfo geometry .pdocu]"

    set chgstatus [info exists DESC_T(changed)]
    set DESC_T(D) \
      [regsub -all { +\n} [.pdocu.outer.info.if.desc_value get 0.0 end] "\n"]
    set DESC_T(D) [regsub {\n$} $DESC_T(D) ""]
    if {!$chgstatus} {
      array unset DESC_T changed
    }

    destroy .pdocu
    Draw $geom
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::Close {} {

  variable DESC
  variable DESC_T

  array unset DESC_T changed
  array set DESC [array get DESC_T]
  set DESC(D) \
    [regsub -all { +\n} [.pdocu.outer.info.if.desc_value get 0.0 end] "\n"]
  set DESC(D) [regsub {\n+$} $DESC(D) ""]

  array unset DESC_T
  destroy .pdocu

}

# ------------------------------------------------------------------------------
proc ::prdoc::Return { wid } {

  if {[winfo class $wid] != "Text"} {
    prdoc::Close
  }

}


# ------------------------------------------------------------------------------
proc ::prdoc::Act { {kill false} }  {

  variable DESC_T

  set rc true

  if {![winfo exists .pdocu]} {return $rc}

  set action "no"
  if {[info exists DESC_T(changed)]} {
    wm deiconify .pdocu
    focus .pdocu
    set action [tk_messageBox -parent .pdocu -icon question -type yesnocancel \
      -default yes -title "[mc menu.prgmdocu]" -message "[mc pdocu.changed]"]
  }

  switch $action {
    "yes" {
      Close
    }
    "no" {
      array unset DESC_T
      if {$kill} {destroy .pdocu}
    }
    "cancel" {
      wm deiconify .pdocu
      focus .pdocu
      set rc false
    }
  }

  return $rc

}

# ------------------------------------------------------------------------------
proc ::prdoc::Purge { prgm } {

  variable DESC
  variable MARKS

  Analyse $prgm
  foreach pd [array names DESC -glob {[LRF]*}] {
    set mm [string index $pd 0]
    if {[lsearch $MARKS($mm) [string range $pd 1 end]] < 0} {
      array unset DESC $pd
    }
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::Changed { wid {n1 ""} {n2 ""} {op ""} } {

  variable DESC_T

  if {![info exists DESC_T(changed)]} {
    set DESC_T(changed) 1
    wm title $wid "[wm title $wid] *"
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::Render { wid } {

  variable TagStyle

  foreach tt [array names TagStyle] {
    lassign $TagStyle($tt) topen tclose topts
    set tname "$topen.*?$tclose"
    set count ""
    set res [$wid search -forwards -regexp -nocase -all -count count -nolinestop \
      $tname 1.0 end]
    if {[llength $res] > 0 && [llength $count] > 0} {
      foreach rr $res cc $count {
        $wid tag add tag$tname $rr "$rr + $cc chars"
      }
    }
    $wid tag configure tag$tname {*}$topts
  }

  set res [$wid search -forwards -regexp -nocase -all -count count \
    "</*\[\[:alnum:]]+\[^>]*>" 1.0 end]
  foreach rr $res cc $count {
    $wid tag add tagtag $rr "$rr + $cc chars"
  }
  $wid tag configure tagtag -elide true

}

# ------------------------------------------------------------------------------
proc ::prdoc::UpdDisplay { wid frm } {

  variable CONF
  variable renderHTML

  if {$renderHTML} {
    prdoc::Render $wid
    $wid configure -state disabled
    set_child_state $frm disabled
  } else {
    $wid tag delete {*}[$wid tag names]
    $wid configure -state normal
    set_child_state $frm !disabled
    if {$CONF(taghighlight)} "HighlightTags $wid"
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::AddTag { wid tag } {

  variable TagStyle

  lassign $TagStyle($tag) topen tclose ign
  set sel [$wid tag ranges sel]
  if {[llength $sel] == 0} {
    $wid insert [$wid index insert] "$topen$tclose"
    $wid mark set insert "[$wid index insert] - [string length $tclose]c"
  } else {
    $wid insert [lindex $sel 1] $tclose
    $wid insert [lindex $sel 0] $topen
  }
  HighlightTags $wid
  focus $wid

}

# ------------------------------------------------------------------------------
proc ::prdoc::HighlightTags { wid } {

  variable CONF

  if {$CONF(taghighlight)} {
    set count {}
    set res [$wid search -forwards -regexp -nocase -all -count count \
      "</*\[\[:alnum:]]+\[^>]*>" 1.0 end]
    foreach rr $res cc $count {
      $wid tag add tagtag $rr "$rr + $cc chars"
    }
    $wid tag configure tagtag -foreground $::HP15(tagcolour)
    if {$::HP15(tagbold)} {
      $wid tag configure tagtag -font FnPrDocstrong
    }
  } else {
    $wid tag configure tagtag -foreground black -font TkTextFont
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::Draw { {geom ""} } {

  variable CONF
  variable LAYOUT
  variable DESC_T
  variable MARKS
  variable renderHTML

  if {[winfo exists .pdocu]} {
    wm deiconify .pdocu
  } else {

    set lrf [expr \
      [llength $MARKS(L)]+[llength $MARKS(R)]+[llength $MARKS(F)]]
    if {[winfo screenheight .] < 801 || $lrf > 30} {
      set colcnt 3.0
      set dhei 5
    } elseif {$lrf > 10} {
      set colcnt 2.0
      set dhei 15
    } else {
      set colcnt 1.0
      set dhei 20
    }
    if {$CONF(ShowResTab)} {
      set dhei 20
    }

    toplevel .pdocu
    wm attributes .pdocu -alpha 0.0

    ttk::frame .pdocu.outer -relief flat

# Program info
    set fpo .pdocu.outer.info
    ttk::labelframe $fpo -relief groove -borderwidth 2 -text " [mc pdocu.description] "

    ttk::frame $fpo.if
    ttk::label $fpo.if.title_label -text [mc pdocu.prgmtitle] -anchor w
    ttk::entry $fpo.if.title_value -textvariable ::prdoc::DESC_T(T)
    grid $fpo.if.title_label -row 0 -column 0 -sticky w
    grid $fpo.if.title_value -row 1 -column 0 -sticky we -columnspan 2

    ttk::label $fpo.if.desc_label -text [mc pdocu.usage] -anchor w
    text $fpo.if.desc_value -width 100 -height $dhei -font TkTextFont -wrap word \
      -undo true -yscrollcommand [list $fpo.if.desc_ysb set]
    if {[tk windowingsystem] == "aqua"} {
      bind $fpo.if.desc_value <Command-a> {%W tag add sel 1.0 end; break;}
    } else {
      bind $fpo.if.desc_value <Control-a> {%W tag add sel 1.0 end; break;}
    }
    ttk::scrollbar $fpo.if.desc_ysb -orient vertical \
      -command [list $fpo.if.desc_value yview]
    grid $fpo.if.desc_label -row 2 -column 0 -sticky w
    grid $fpo.if.desc_value -row 3 -column 0 -sticky nwse -columnspan 7
    grid $fpo.if.desc_ysb -row 3 -column 7 -sticky ns

    ttk::frame $fpo.if.tags
    ttk::button $fpo.if.tags.bold -text "B" -width $LAYOUT(btnwid2) \
      -style strong.TButton -command "prdoc::AddTag $fpo.if.desc_value strong"
    ttk::button $fpo.if.tags.italic -text "I" -width $LAYOUT(btnwid2) \
      -style em.TButton -command "prdoc::AddTag $fpo.if.desc_value em"
    ttk::button $fpo.if.tags.code -text "code" -width 5 \
      -command "prdoc::AddTag $fpo.if.desc_value code"
    ttk::label $fpo.if.tags.sep1 -text "" -width 1

    ttk::button $fpo.if.tags.ul -text "ul" -width $LAYOUT(btnwid2) \
      -command "prdoc::AddTag $fpo.if.desc_value ul"
    ttk::button $fpo.if.tags.li -text "li" -width $LAYOUT(btnwid2) \
      -command "prdoc::AddTag $fpo.if.desc_value li"
    ttk::label $fpo.if.tags.sep2 -text "" -width 1

    ttk::button $fpo.if.tags.reg -text "X" -width $LAYOUT(btnwid2) \
      -style reg.TButton -command "prdoc::AddTag $fpo.if.desc_value register"
    ttk::button $fpo.if.tags.keylbl -text "123" -width $LAYOUT(btnwid2) \
      -style strong.TButton -command "prdoc::AddTag $fpo.if.desc_value KeyLabel"
    ttk::button $fpo.if.tags.fkeylbl -text "FFF" -width $LAYOUT(btnwid2) \
      -style fkey.TButton -command "prdoc::AddTag $fpo.if.desc_value fKeyLabel"
    ttk::button $fpo.if.tags.gkeylbl -text "GGG" -width $LAYOUT(btnwid2) \
      -style gkey.TButton -command "prdoc::AddTag $fpo.if.desc_value gKeyLabel"
    ttk::button $fpo.if.tags.f -text "f" -width $LAYOUT(btnwid2) \
      -style fkey.TButton -command "prdoc::AddTag $fpo.if.desc_value fKey"
    ttk::button $fpo.if.tags.g -text "g" -width $LAYOUT(btnwid2) \
      -style gkey.TButton -command "prdoc::AddTag $fpo.if.desc_value gKey"
    ttk::label $fpo.if.tags.sep3 -text "" -width 1

    ttk::checkbutton $fpo.if.tags.hltags -text [mc pdocu.hilitags] \
      -variable ::prdoc::CONF(taghighlight) -command "prdoc::HighlightTags $fpo.if.desc_value "

    grid $fpo.if.tags.bold -row 0 -column 0 -sticky nw
    grid $fpo.if.tags.italic -row 0 -column 1 -sticky nw
    grid $fpo.if.tags.code -row 0 -column 2 -sticky nw
    grid $fpo.if.tags.sep1 -row 0 -column 3 -sticky nw
    grid $fpo.if.tags.ul -row 0 -column 4 -sticky nw
    grid $fpo.if.tags.li -row 0 -column 5 -sticky nw
    grid $fpo.if.tags.sep2 -row 0 -column 6 -sticky nw
    grid $fpo.if.tags.reg -row 0 -column 7 -sticky nw
    grid $fpo.if.tags.keylbl -row 0 -column 8 -sticky nw
    grid $fpo.if.tags.fkeylbl -row 0 -column 9 -sticky nw
    grid $fpo.if.tags.gkeylbl -row 0 -column 10 -sticky nw
    grid $fpo.if.tags.f -row 0 -column 11 -sticky nw
    grid $fpo.if.tags.g -row 0 -column 12 -sticky nw
    grid $fpo.if.tags.sep3 -row 0 -column 13 -sticky nw
    grid $fpo.if.tags.hltags -row 0 -column 14 -sticky wse
    grid $fpo.if.tags -row 4 -column 0 -sticky nw

    ttk::checkbutton $fpo.if.render -text [mc pdocu.render_html] \
      -variable prdoc::renderHTML \
      -command "prdoc::UpdDisplay $fpo.if.desc_value $fpo.if.tags"
    grid $fpo.if.render -row 4 -column 1 -sticky se

    ttk::frame $fpo.if.filler -height 5
    grid $fpo.if.filler -row 5 -column 0 -sticky we
    grid columnconfigure $fpo.if 0 -weight 1
    grid rowconfigure $fpo.if 3 -weight 1
    grid $fpo.if -row 0 -column 0 -padx 10 -sticky nwse

    grid columnconfigure $fpo 0 -weight 1
    grid rowconfigure $fpo 0 -weight 1

    if {[info exists DESC_T(D)]} {
      $fpo.if.desc_value insert 0.0 $DESC_T(D)
      {*}[$fpo.if.render cget -command]
    }

# Tabbed?
    if {$CONF(ShowResTab)} {
      set resfrm .pdocu.outer.tabs
      ttk::notebook $resfrm -padding [list 3 3 2 0]
    } else {
      set resfrm .pdocu.outer
    }

# Labels
    if {$CONF(ShowResTab)} {
      set flbl $resfrm.lbl
    } else {
      ttk::labelframe $resfrm.lbl -relief groove -borderwidth 2 -text " [mc gen.labels] "
      grid columnconfigure $resfrm.lbl 0 -weight 1
      set flbl $resfrm.lbl.if
    }
    ttk::frame $flbl

    set rr 0
    set cc 0
    foreach ll $MARKS(L) {
      ttk::label $flbl.label$ll -text "[format_mark $ll] " -width 3 -anchor e
      ttk::entry $flbl.value$ll -width 40 -textvariable prdoc::DESC_T(L$ll)
      grid $flbl.label$ll -row $rr -column $cc -sticky e -pady 1
      grid $flbl.value$ll -row $rr -column [expr $cc+1] -sticky we -pady 1
      incr rr
      if {$rr > int(ceil([llength $MARKS(L)]/$colcnt))-1} {
        set rr 0
        incr cc 2
      }
    }
    if {[llength $MARKS(L)] == 0} {
      ttk::label $flbl.nolabels -text [mc pdocu.nolabels] -anchor w -justify left
      grid $flbl.nolabels -row 0 -column 0 -sticky w -pady 1
    }
    if {[llength $MARKS(L)] == 1 || $colcnt == 1} {
      grid columnconfigure $flbl {1} -weight 2
    } else {
      for {set ii 1} {$ii < $colcnt*2} {incr ii 2} {
        grid columnconfigure $flbl $ii -weight 2
      }
    }
    grid $flbl -row 0 -column 0 -padx 10 -pady 5 -sticky nwse

# Registers
    if {$CONF(ShowResTab)} {
      set fregs $resfrm.regs
    } else {
      ttk::labelframe $resfrm.regs -relief groove -borderwidth 2 \
        -text " [mc gen.storageregs] "
      grid columnconfigure $resfrm.regs 0 -weight 1
      set fregs $resfrm.regs.if
    }
    ttk::frame $fregs

    set rr 0
    set cc 0
    foreach ll $MARKS(R) {
      ttk::label $fregs.label$ll -text "[format_mark $ll] " -width 3 -anchor e
      ttk::entry $fregs.value$ll -width 40 -textvariable prdoc::DESC_T(R$ll)
      grid $fregs.label$ll -row $rr -column $cc -sticky e -pady 1
      grid $fregs.value$ll -row $rr -column [expr $cc+1] -sticky we -pady 1
      incr rr
      if {$rr > int(ceil([llength $MARKS(R)]/$colcnt))-1} {
        set rr 0
        incr cc 2
      }
    }
    if {[llength $MARKS(R)] == 0} {
      ttk::label $fregs.noregs -text [mc pdocu.noregs] -anchor w -justify left
      grid $fregs.noregs -row 0 -column 0 -sticky w -pady 1
    }
    if {[llength $MARKS(R)] == 1 || $colcnt == 1} {
      grid columnconfigure $fregs {1} -weight 2
    } else {
      for {set ii 1} {$ii < $colcnt*2} {incr ii 2} {
        grid columnconfigure $fregs $ii -weight 2
      }
    }
    grid $fregs -row 0 -column 0 -padx 10 -pady 5 -sticky nwse

# Flags
    if {$CONF(ShowResTab)} {
      set fflags $resfrm.flags
    } else {
      ttk::labelframe $resfrm.flags -relief groove -borderwidth 2 \
        -text " [mc gen.flags] "
      grid columnconfigure $resfrm.flags 0 -weight 1
      set fflags $resfrm.flags.if
    }
    ttk::frame $fflags

    set rr 0
    set cc 0
    foreach ll $MARKS(F) {
      ttk::label $fflags.label$ll -text " $ll " -width 3 -anchor e -justify right
      ttk::entry $fflags.value$ll -width 40 -textvariable prdoc::DESC_T(F$ll)
      grid $fflags.label$ll -row $rr -column $cc -sticky e -pady 1
      grid $fflags.value$ll -row $rr -column [expr $cc+1] -sticky we -pady 1
      incr rr
      if {$rr > int(ceil([llength $MARKS(F)]/$colcnt))-1} {
        set rr 0
        incr cc 2
      }
    }
    if {[llength $MARKS(F)] == 0} {
      ttk::label $fflags.noflags -text [mc pdocu.noflags] -anchor w -justify left
      grid $fflags.noflags -row 0 -column 0 -sticky w -pady 1
    }
    if {[llength $MARKS(F)] == 1 || $colcnt == 1} {
      grid columnconfigure $fflags {1} -weight 2
    } else {
      for {set ii 1} {$ii < $colcnt*2} {incr ii 2} {
        grid columnconfigure $fflags $ii -weight 2
      }
    }
    grid $fflags -row 0 -column 0 -padx 10 -pady 5 -sticky nwse

# Lay out dialogue
    set fpo .pdocu.outer
    grid $fpo.info -row 0 -column 0 -sticky nsew -padx 3 -pady 3
    if {$CONF(ShowResTab)} {
      foreach tab [list $flbl $fregs $fflags] {
        $tab configure -padding [list 3 10 10 3]
      }
      $resfrm add $flbl -text " [mc gen.labels] " -sticky nsew
      $resfrm add $fregs -text " [mc gen.storageregs] " -sticky nsew
      $resfrm add $fflags -text " [mc gen.flags] " -sticky nsew
      grid $resfrm -row 1 -column 0 -sticky nsew
    } else {
      grid $resfrm.lbl -row 1 -column 0 -sticky nsew -padx 3
      grid $resfrm.regs -row 2 -column 0 -sticky nsew -padx 3 -pady 3
      grid $resfrm.flags -row 3 -column 0 -sticky nsew -padx 3
    }
    grid columnconfigure $fpo 0 -weight 1
    grid rowconfigure $fpo 0 -weight 1

    grid $fpo -row 0 -column 0 -sticky nsew

# Button frame
    set fbtn .pdocu.btn
    ttk::frame $fbtn -relief flat -borderwidth 8
    ttk::checkbutton $fbtn.restab -text "[mc pdocu.restab]" \
      -variable ::prdoc::CONF(ShowResTab) -command "prdoc::ReDraw"
    ttk::button $fbtn.reload -text [mc pdocu.reload] -command "prdoc::Reload"
    ttk::button $fbtn.ok -text [mc gen.ok] -command "prdoc::Close" -default active
    ttk::button $fbtn.cancel -text [mc gen.cancel] -command "prdoc::Act true"

    grid $fbtn.restab -row 0 -column 0 -padx 5 -pady 0 -sticky w
    grid $fbtn.reload -row 0 -column 1 -padx 20 -pady 0 -sticky e
    grid $fbtn.ok -row 0 -column 2 -padx 5 -pady 0 -sticky e
    grid $fbtn.cancel -row 0 -column 3 -padx 5 -pady 0 -sticky e
    grid $fbtn -row 1 -column 0 -sticky nsew
    grid columnconfigure $fbtn 0 -weight 1

    bind .pdocu <Return> "prdoc::Return %W"
    bind .pdocu <Escape> "$fbtn.cancel invoke"

    grid columnconfigure .pdocu 0 -weight 1
    grid rowconfigure .pdocu 0 -weight 1

    if {$::HP15(prgmname) != ""} {
      wm title .pdocu "$::APPDATA(appname) [mc gen.program]: $::HP15(prgmname)"
    } else {
      wm title .pdocu "$::APPDATA(appname) [mc gen.program]: [mc pdocu.notsaved]"
    }

    wm minsize .pdocu [expr int([winfo width .pdocu]*0.67)] \
      [expr int([winfo height .pdocu]*0.85)]

# Track changes on documentation
    .pdocu.outer.info.if.desc_value edit modified 0
    update
    bind .pdocu.outer.info.if.desc_value <<Modified>> "prdoc::Changed .pdocu"
    if {[trace info variable prdoc::DESC_T] == ""} {
      trace add variable prdoc::DESC_T write "prdoc::Changed .pdocu"
    }
    wm protocol .pdocu WM_DELETE_WINDOW "prdoc::Act true"

    if {$geom != ""} {
      wm geometry .pdocu $geom
    }
    wm attributes .pdocu -alpha 1.0
    focus .pdocu.outer.info.if.title_value
    raise .pdocu
  }

}

# ------------------------------------------------------------------------------
proc ::prdoc::Edit { {geom ""} } {

  variable DESC
  variable DESC_T

  array unset DESC_T
  array set DESC_T [array get DESC]

  Analyse $::PRGM
  Draw $geom

}
