/*
 * XML Catalog Manager (xmlcatmgr)
 * $Id: mem.c,v 1.1 2004/08/31 19:07:23 jmmv Exp $
 *
 * Copyright (c) 2003, 2004 Julio M. Merino Vidal.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the author nor the names of contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * This file implements several functions that act as wrappers around
 * their respective system function.  They take care to count how many
 * memory blocks are allocated at a time so that we can detect memory
 * leaks.  They also check if there was enough memory to allocate a
 * required chunk and print a common warning message in case of error.
 */

#include "system.h"

#ifndef lint
__RCSID("$Id: mem.c,v 1.1 2004/08/31 19:07:23 jmmv Exp $");
#endif

#define MEM_NOT_OVERRIDE
#include "mem.h"

static long Balance = 0;

/* --------------------------------------------------------------------- */

void *
mem_malloc(size_t size)
{
    void *ptr;

    ptr = malloc(size);
    if (ptr == NULL)
        warnx("not enough memory to allocate %zd bytes", size);
    else
        Balance++;

    return ptr;
}

/* --------------------------------------------------------------------- */

void
mem_free(void *ptr)
{
    Balance--;

    free(ptr);
}

/* --------------------------------------------------------------------- */

char *
mem_strdup(const char *str)
{
    char *ptr;

    ptr = strdup(str);
    if (ptr == NULL)
        warnx("not enough memory to duplicate string `%s'", str);
    else
        Balance++;

    return ptr;
}

/* --------------------------------------------------------------------- */

void
mem_status(const char *catname, int argc, char *const *argv)
{
    if (Balance > 0) {
        int i;

        fflush(stdout);

        fprintf(stderr,
            "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n"
            "!!!!!               XMLCATMGR - MEMORY LEAK DETECTED             !!!!!\n"
            "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n"
            "xmlcatmgr has detected a memory leak during operation.  This means\n"
            "that you have hit a (non-fatal) bug in the software.  Even though,\n"
            "note that the program has finished successfully.\n\n"
            "Please, consider reporting this bug back to the author, so that it\n"
            "can be fixed in the next version.  An effective bug report against\n"
            "this issue should include a copy of the catalog file your were\n"
            "workig on and the command line used to execute the program; all\n"
            "these details are shown below for your convenience.\n\n"
            "Memory balance: %ld\n"
            "Catalog file: %s\n"
            "Command line:", Balance, catname);

        for (i = 0; i < argc; i++)
            fprintf(stderr, " %s", argv[i]);

        fprintf(stderr,
            "\n\n"
            "Send these details to <jmmv@users.sourceforge.net>, please.\n"
            "Thank you.\n\n"
            "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
    }
}

/*
 * Local Variables: ***
 * mode: c ***
 * c-file-style: "stroustrup" ***
 * End: ***
 * vim: syntax=c:expandtab:shiftwidth=4:softtabstop=4
 */
